/**  @file bta_morphology.c
*  
*    @brief This file implements some morphology filters
*  
*    BLT_DISCLAIMER
*  
*    @author Alex Falkensteiner
*  
*    @cond svn
*  
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*  
*    @endcond
*/


#include "bta_morphology.h"
#include <stdlib.h>
#include <string.h>

#ifndef BTA_EXCLUDE_FILTERS


// Local prototypes
static BTA_Status erosion(BTA_FltMorphologyInst *inst, BTA_Channel **channel);
static BTA_Status dilation(BTA_FltMorphologyInst *inst, BTA_Channel **channel);

BTA_Status BFLTmorphologyInit(BTA_FltMorphologyConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    BTA_FltMorphologyInst *inst;
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    inst = (BTA_FltMorphologyInst *)calloc(1, sizeof(BTA_FltMorphologyInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->infoEventInst = infoEventInst;
    inst->morphologyType = config->morphologyType;
    inst->channelToProcess = config->channelToProcess;
    if (config->xRes % 2 != 1 || config->yRes % 2 != 1) {
        BTAinfoEventHelper(infoEventInst, 1, BTA_StatusInvalidParameter, "BFLTmorphologyInit: The mask must be of uneven width and height", 0);
        return BTA_StatusInvalidParameter;
    }
    inst->mask = (uint8_t *)malloc(config->xRes * config->yRes * sizeof(uint8_t));
    if (!inst->mask) {
        free(inst);
        return BTA_StatusOutOfMemory;
    }
    memcpy(inst->mask, config->mask, config->xRes * config->yRes * sizeof(uint8_t));
    inst->xRes = config->xRes;
    inst->yRes = config->yRes;
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTmorphologyClose(BTA_FltHandle *handle) {
    BTA_FltMorphologyInst **inst;
    inst = (BTA_FltMorphologyInst **)handle;
    free((*inst)->mask);
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTmorphologyApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_Status status;
    BTA_FltMorphologyInst *inst;
    int chIn;
    inst = (BTA_FltMorphologyInst *)handle;
    if (!inst || !frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!*frame) {
        return BTA_StatusInvalidParameter;
    }
    for (chIn = 0; chIn < (*frame)->channelsLen; chIn++) {
        if ((*frame)->channels[chIn]->id == inst->channelToProcess) {
            switch (inst->morphologyType) {
            case BTA_FltMorphologyTypeErosion:
                status = erosion(inst, &((*frame)->channels[chIn]));
                if (status != BTA_StatusOk) {
                    return status;
                }
                break;
            case BTA_FltMorphologyTypeDilation:
                status = dilation(inst, &((*frame)->channels[chIn]));
                if (status != BTA_StatusOk) {
                    return status;
                }
                break;
            default:
                            //infoEventHelper
                return BTA_StatusRuntimeError;
            }
        }
    }
    return BTA_StatusOk;
}


static BTA_Status erosion(BTA_FltMorphologyInst *inst, BTA_Channel **channel) {
    BTA_Status status;
    BTA_Channel *channelNew, *channelTemp;
    int x, y, xm, ym, halfX, halfY;
    uint8_t filled;
    if (!inst || !channel) {
        return BTA_StatusInvalidParameter;
    }
    if (!*channel) {
        return BTA_StatusInvalidParameter;
    }
    channelTemp = *channel;
    *channel = 0;
    status = BTAcloneChannel(channelTemp, &channelNew);
    if (status != BTA_StatusOk) {
        return status;
    }
    memset(channelNew->data, 0, channelNew->dataLen);
    halfX = inst->xRes / 2;
    halfY = inst->yRes / 2;
    for (y = halfY; y < channelTemp->yRes - halfY; y++) {
        for (x = halfX; x < channelTemp->xRes - halfX; x++) {
            filled = 1;
            for (ym = -halfY; ym <= halfY; ym++) {
                for (xm = -halfX; xm <= halfX; xm++) {
                    switch (channelTemp->dataFormat) {
                    case BTA_DataFormatUInt16:
                        if (!((uint16_t *)channelTemp->data)[x+xm+(y+ym)*channelTemp->xRes] && inst->mask[xm+halfX+(ym+halfY)*inst->xRes]) {
                            filled = 0;
                        }
                        break;
                    default:
                            //infoEventHelper
                        break;
                    }
                    if (!filled) {
                        break;
                    }
                }
                if (!filled) {
                    break;
                }
            }
            if (filled) {
                switch (channelTemp->dataFormat) {
                case BTA_DataFormatUInt16:
                    ((uint16_t *)channelNew->data)[x + y*channelNew->xRes] = 1;
                    break;
                default:
                    break;
                }
            }
        }
    }
    BTAfreeChannel(&channelTemp);
    *channel = channelNew;
    return BTA_StatusOk;
}


static BTA_Status dilation(BTA_FltMorphologyInst *inst, BTA_Channel **channel) {
    BTA_Status status;
    BTA_Channel *channelNew, *channelTemp;
    int x, y, xm, ym, halfX, halfY;
    uint8_t filled;
    if (!inst || !channel) {
        return BTA_StatusInvalidParameter;
    }
    if (!*channel) {
        return BTA_StatusInvalidParameter;
    }
    channelTemp = *channel;
    *channel = 0;
    status = BTAcloneChannel(channelTemp, &channelNew);
    if (status != BTA_StatusOk) {
        return status;
    }
    memset(channelNew->data, 0, channelNew->dataLen);
    halfX = inst->xRes / 2;
    halfY = inst->yRes / 2;
    for (y = halfY; y < channelTemp->yRes - halfY; y++) {
        for (x = halfX; x < channelTemp->xRes - halfX; x++) {
            switch (channelTemp->dataFormat) {
            case BTA_DataFormatUInt16:
                filled = ((uint16_t *)channelTemp->data)[x + y*channelTemp->xRes] > 0;
                break;
            default:
            filled = 0;
                //infoEventHelper
                break;
            }
            if (filled) {
                for (ym = -halfY; ym <= halfY; ym++) {
                    for (xm = -halfX; xm <= halfX; xm++) {
                        switch (channelTemp->dataFormat) {
                        case BTA_DataFormatUInt16:
                            if (inst->mask[xm+halfX + (ym + halfY)*inst->xRes]) {
                                ((uint16_t *)channelNew->data)[x+xm+(y+ym)*channelNew->xRes] = 1;
                            }
                            break;
                        default:
                            //infoEventHelper
                            break;
                        }
                    }
                }
            }
        }
    }
    BTAfreeChannel(&channelTemp);
    *channel = channelNew;
    return BTA_StatusOk;
}
#endif
